/**
 * CONEXUS OMS - Data & Helpers Module
 * Contains all data and helper functions
 */

// ==========================================
// STATIC DATA
// ==========================================

const ConexusData = {
    // Configuration
    config: {
        statuses: ['Placed', 'Confirmed', 'Production', 'Cargo Ready', 'Shipped', 'In Transit', 'Arrived', 'Delivered', 'On Hold', 'Cancelled'],
        modes: ['Sea FCL', 'Sea LCL', 'Air Freight', 'Road', 'Rail', 'Multimodal'],
        incoterms: ['FOB', 'CIF', 'EXW', 'DDP', 'DAP', 'CFR', 'CIP', 'FCA'],
        priorities: ['Critical', 'High', 'Medium', 'Low'],
        alertTypes: ['Delay', 'Missing Document', 'Milestone Overdue', 'Data Mismatch', 'Compliance', 'Capacity', 'Weather', 'Port Congestion']
    },

    // Sample Orders
    orders: [
        {
            id: 'PO-2025-1847',
            poNumber: 'PO-2025-1847',
            buyerId: 'BUY001',
            supplierId: 'SUP001',
            forwarderId: 'FWD001',
            originAgentId: 'AGT001',
            destAgentId: 'AGT002',
            status: 'In Transit',
            mode: 'Sea FCL',
            incoterm: 'FOB',
            origin: 'Yantian, China',
            destination: 'Los Angeles, USA',
            cargoReadyDate: '2025-01-02',
            eta: '2025-01-25',
            etaConfidence: 'Confirmed',
            vesselName: 'COSCO Shipping Aries',
            voyageNumber: 'AE2501W',
            containerNumber: 'CSNU7234567',
            blNumber: 'COSU6278943215',
            totalValue: 245000,
            currency: 'USD',
            totalWeight: 18500,
            weightUnit: 'kg',
            dataQuality: 92,
            priority: 'High',
            notes: 'Priority shipment for Q1 launch',
            createdAt: '2024-12-15T08:30:00Z',
            updatedAt: '2025-01-08T10:15:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-12-15', actual: '2024-12-15' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-12-16', actual: '2024-12-16' },
                { name: 'Production Started', status: 'completed', date: '2024-12-18', actual: '2024-12-18' },
                { name: 'Production Complete', status: 'completed', date: '2024-12-28', actual: '2024-12-28' },
                { name: 'Cargo Ready', status: 'completed', date: '2025-01-02', actual: '2025-01-02' },
                { name: 'Shipped', status: 'completed', date: '2025-01-05', actual: '2025-01-05' },
                { name: 'Port Arrival', status: 'pending', date: '2025-01-22', actual: null },
                { name: 'Customs Cleared', status: 'pending', date: '2025-01-24', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-01-25', actual: null }
            ],
            lineItems: [
                { sku: 'ELEC-TV-55', description: '55" Smart LED TV', qty: 200, unitPrice: 450, total: 90000 },
                { sku: 'ELEC-TV-65', description: '65" Smart LED TV', qty: 150, unitPrice: 680, total: 102000 },
                { sku: 'ELEC-ACC-01', description: 'Wall Mount Kit', qty: 350, unitPrice: 45, total: 15750 },
                { sku: 'ELEC-ACC-02', description: 'Extended Warranty Card', qty: 350, unitPrice: 25, total: 8750 },
                { sku: 'ELEC-REMOTE', description: 'Smart Remote', qty: 350, unitPrice: 82, total: 28500 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Received', date: '2025-01-03' },
                { name: 'Packing List', type: 'Packing', status: 'Received', date: '2025-01-03' },
                { name: 'Bill of Lading', type: 'BL', status: 'Received', date: '2025-01-05' },
                { name: 'Certificate of Origin', type: 'COO', status: 'Pending', date: null },
                { name: 'Insurance Certificate', type: 'Insurance', status: 'Received', date: '2025-01-04' }
            ],
            messages: [
                { from: 'Supplier', text: 'Production completed ahead of schedule. Cargo ready for pickup.', date: '2024-12-28T14:20:00Z' },
                { from: 'Agent', text: 'Container loaded. Vessel departed Yantian on schedule.', date: '2025-01-05T18:45:00Z' },
                { from: 'System', text: 'ETA updated based on vessel tracking.', date: '2025-01-08T09:00:00Z' }
            ],
            co2Emissions: 2.8
        },
        {
            id: 'PO-2025-1846',
            poNumber: 'PO-2025-1846',
            buyerId: 'BUY002',
            supplierId: 'SUP002',
            forwarderId: 'FWD002',
            originAgentId: 'AGT003',
            destAgentId: 'AGT004',
            status: 'Production',
            mode: 'Air Freight',
            incoterm: 'CIF',
            origin: 'Singapore',
            destination: 'London, UK',
            cargoReadyDate: '2025-01-15',
            eta: '2025-01-18',
            etaConfidence: 'Estimated',
            vesselName: null,
            voyageNumber: null,
            containerNumber: null,
            blNumber: null,
            flightNumber: 'SQ322',
            awbNumber: 'AWBSG2501846',
            totalValue: 890000,
            currency: 'USD',
            totalWeight: 2500,
            weightUnit: 'kg',
            dataQuality: 78,
            priority: 'Critical',
            notes: 'Urgent automotive parts - production line dependency',
            createdAt: '2024-12-20T11:00:00Z',
            updatedAt: '2025-01-07T16:30:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-12-20', actual: '2024-12-20' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-12-21', actual: '2024-12-22' },
                { name: 'Production Started', status: 'completed', date: '2025-01-03', actual: '2025-01-03' },
                { name: 'Production Complete', status: 'in-progress', date: '2025-01-12', actual: null },
                { name: 'Cargo Ready', status: 'pending', date: '2025-01-15', actual: null },
                { name: 'Shipped', status: 'pending', date: '2025-01-16', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-01-18', actual: null }
            ],
            lineItems: [
                { sku: 'AUTO-ENG-001', description: 'Engine Control Module', qty: 500, unitPrice: 1200, total: 600000 },
                { sku: 'AUTO-SEN-015', description: 'Oxygen Sensor Assembly', qty: 1000, unitPrice: 180, total: 180000 },
                { sku: 'AUTO-CON-003', description: 'Wiring Harness Set', qty: 500, unitPrice: 220, total: 110000 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Draft', date: null },
                { name: 'Packing List', type: 'Packing', status: 'Pending', date: null },
                { name: 'Air Waybill', type: 'AWB', status: 'Pending', date: null }
            ],
            messages: [
                { from: 'Supplier', text: 'Production at 65%. On track for Jan 12 completion.', date: '2025-01-07T10:00:00Z' }
            ],
            co2Emissions: 8.2
        },
        {
            id: 'PO-2025-1845',
            poNumber: 'PO-2025-1845',
            buyerId: 'BUY003',
            supplierId: 'SUP003',
            forwarderId: 'FWD001',
            originAgentId: 'AGT001',
            destAgentId: 'AGT002',
            status: 'Cargo Ready',
            mode: 'Sea FCL',
            incoterm: 'FOB',
            origin: 'Kaohsiung, Taiwan',
            destination: 'Rotterdam, Netherlands',
            cargoReadyDate: '2025-01-06',
            eta: '2025-02-08',
            etaConfidence: 'Estimated',
            vesselName: 'MSC Oscar',
            voyageNumber: 'EU2502E',
            containerNumber: 'MSCU8876543',
            blNumber: null,
            totalValue: 567000,
            currency: 'USD',
            totalWeight: 22000,
            weightUnit: 'kg',
            dataQuality: 65,
            priority: 'Medium',
            notes: 'Semiconductor components - handle with care',
            createdAt: '2024-12-10T09:15:00Z',
            updatedAt: '2025-01-06T14:20:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-12-10', actual: '2024-12-10' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-12-11', actual: '2024-12-12' },
                { name: 'Production Started', status: 'completed', date: '2024-12-15', actual: '2024-12-15' },
                { name: 'Production Complete', status: 'completed', date: '2025-01-04', actual: '2025-01-05' },
                { name: 'Cargo Ready', status: 'completed', date: '2025-01-06', actual: '2025-01-06' },
                { name: 'Shipped', status: 'pending', date: '2025-01-10', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-02-08', actual: null }
            ],
            lineItems: [
                { sku: 'SEMI-IC-500', description: 'Integrated Circuit Module', qty: 10000, unitPrice: 45, total: 450000 },
                { sku: 'SEMI-CAP-200', description: 'Capacitor Array', qty: 5000, unitPrice: 18, total: 90000 },
                { sku: 'SEMI-RES-100', description: 'Resistor Kit', qty: 5000, unitPrice: 5.4, total: 27000 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Received', date: '2025-01-06' },
                { name: 'Packing List', type: 'Packing', status: 'Received', date: '2025-01-06' },
                { name: 'Bill of Lading', type: 'BL', status: 'Pending', date: null }
            ],
            messages: [
                { from: 'Supplier', text: 'Cargo ready at warehouse. Awaiting vessel booking confirmation.', date: '2025-01-06T12:00:00Z' }
            ],
            co2Emissions: 3.1
        },
        {
            id: 'PO-2025-1844',
            poNumber: 'PO-2025-1844',
            buyerId: 'BUY001',
            supplierId: 'SUP004',
            forwarderId: 'FWD003',
            originAgentId: 'AGT005',
            destAgentId: 'AGT002',
            status: 'Shipped',
            mode: 'Sea LCL',
            incoterm: 'DAP',
            origin: 'Tokyo, Japan',
            destination: 'New York, USA',
            cargoReadyDate: '2024-12-28',
            eta: '2025-01-20',
            etaConfidence: 'Confirmed',
            vesselName: 'ONE Commitment',
            voyageNumber: 'PS2501W',
            containerNumber: 'ONEU5543210',
            blNumber: 'ONEYJTK12345',
            totalValue: 178000,
            currency: 'USD',
            totalWeight: 4200,
            weightUnit: 'kg',
            dataQuality: 88,
            priority: 'Medium',
            notes: 'Precision instruments - temperature controlled',
            createdAt: '2024-12-05T14:30:00Z',
            updatedAt: '2025-01-08T08:00:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-12-05', actual: '2024-12-05' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-12-06', actual: '2024-12-06' },
                { name: 'Production Started', status: 'completed', date: '2024-12-10', actual: '2024-12-10' },
                { name: 'Production Complete', status: 'completed', date: '2024-12-26', actual: '2024-12-26' },
                { name: 'Cargo Ready', status: 'completed', date: '2024-12-28', actual: '2024-12-28' },
                { name: 'Shipped', status: 'completed', date: '2025-01-02', actual: '2025-01-02' },
                { name: 'Port Arrival', status: 'pending', date: '2025-01-18', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-01-20', actual: null }
            ],
            lineItems: [
                { sku: 'INST-CAL-01', description: 'Calibration Equipment Set', qty: 50, unitPrice: 2400, total: 120000 },
                { sku: 'INST-MET-02', description: 'Precision Meter', qty: 100, unitPrice: 580, total: 58000 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Received', date: '2024-12-30' },
                { name: 'Packing List', type: 'Packing', status: 'Received', date: '2024-12-30' },
                { name: 'Bill of Lading', type: 'BL', status: 'Received', date: '2025-01-02' },
                { name: 'Certificate of Origin', type: 'COO', status: 'Received', date: '2024-12-30' }
            ],
            messages: [],
            co2Emissions: 1.9
        },
        {
            id: 'PO-2025-1843',
            poNumber: 'PO-2025-1843',
            buyerId: 'BUY004',
            supplierId: 'SUP005',
            forwarderId: 'FWD002',
            originAgentId: 'AGT001',
            destAgentId: 'AGT006',
            status: 'On Hold',
            mode: 'Sea FCL',
            incoterm: 'CIF',
            origin: 'Shanghai, China',
            destination: 'Hamburg, Germany',
            cargoReadyDate: '2025-01-08',
            eta: '2025-02-15',
            etaConfidence: 'Unknown',
            vesselName: null,
            voyageNumber: null,
            containerNumber: null,
            blNumber: null,
            totalValue: 423000,
            currency: 'USD',
            totalWeight: 28000,
            weightUnit: 'kg',
            dataQuality: 45,
            priority: 'High',
            notes: 'ON HOLD - Payment verification pending',
            createdAt: '2024-12-18T10:00:00Z',
            updatedAt: '2025-01-07T11:30:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-12-18', actual: '2024-12-18' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-12-19', actual: '2024-12-20' },
                { name: 'Production Started', status: 'completed', date: '2024-12-22', actual: '2024-12-23' },
                { name: 'Production Complete', status: 'completed', date: '2025-01-06', actual: '2025-01-06' },
                { name: 'Cargo Ready', status: 'on-hold', date: '2025-01-08', actual: null },
                { name: 'Shipped', status: 'pending', date: '2025-01-15', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-02-15', actual: null }
            ],
            lineItems: [
                { sku: 'TEXT-FAB-100', description: 'Cotton Fabric Roll', qty: 500, unitPrice: 340, total: 170000 },
                { sku: 'TEXT-FAB-200', description: 'Polyester Blend Roll', qty: 400, unitPrice: 420, total: 168000 },
                { sku: 'TEXT-ACC-050', description: 'Button & Zipper Kit', qty: 1000, unitPrice: 85, total: 85000 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Draft', date: null },
                { name: 'Letter of Credit', type: 'LC', status: 'Pending Verification', date: null }
            ],
            messages: [
                { from: 'Finance', text: 'Order on hold pending LC verification. Expected resolution by Jan 10.', date: '2025-01-07T11:30:00Z' }
            ],
            co2Emissions: null
        },
        {
            id: 'PO-2025-1842',
            poNumber: 'PO-2025-1842',
            buyerId: 'BUY005',
            supplierId: 'SUP001',
            forwarderId: 'FWD001',
            originAgentId: 'AGT001',
            destAgentId: 'AGT002',
            status: 'Delivered',
            mode: 'Sea FCL',
            incoterm: 'FOB',
            origin: 'Yantian, China',
            destination: 'Long Beach, USA',
            cargoReadyDate: '2024-12-01',
            eta: '2024-12-28',
            etaConfidence: 'Confirmed',
            vesselName: 'Evergreen Ever Given',
            voyageNumber: 'AW2451E',
            containerNumber: 'EGHU4523890',
            blNumber: 'EGLV123456789',
            totalValue: 312000,
            currency: 'USD',
            totalWeight: 15800,
            weightUnit: 'kg',
            dataQuality: 95,
            priority: 'Low',
            notes: 'Completed order - all documents received',
            createdAt: '2024-11-10T08:00:00Z',
            updatedAt: '2024-12-30T14:00:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2024-11-10', actual: '2024-11-10' },
                { name: 'PO Confirmed', status: 'completed', date: '2024-11-11', actual: '2024-11-11' },
                { name: 'Production Started', status: 'completed', date: '2024-11-15', actual: '2024-11-15' },
                { name: 'Production Complete', status: 'completed', date: '2024-11-28', actual: '2024-11-28' },
                { name: 'Cargo Ready', status: 'completed', date: '2024-12-01', actual: '2024-12-01' },
                { name: 'Shipped', status: 'completed', date: '2024-12-05', actual: '2024-12-05' },
                { name: 'Port Arrival', status: 'completed', date: '2024-12-26', actual: '2024-12-26' },
                { name: 'Customs Cleared', status: 'completed', date: '2024-12-27', actual: '2024-12-27' },
                { name: 'Delivered', status: 'completed', date: '2024-12-28', actual: '2024-12-28' }
            ],
            lineItems: [
                { sku: 'HOME-FUR-001', description: 'Office Chair', qty: 500, unitPrice: 420, total: 210000 },
                { sku: 'HOME-FUR-002', description: 'Standing Desk', qty: 200, unitPrice: 510, total: 102000 }
            ],
            documents: [
                { name: 'Commercial Invoice', type: 'Invoice', status: 'Received', date: '2024-12-02' },
                { name: 'Packing List', type: 'Packing', status: 'Received', date: '2024-12-02' },
                { name: 'Bill of Lading', type: 'BL', status: 'Received', date: '2024-12-05' },
                { name: 'Certificate of Origin', type: 'COO', status: 'Received', date: '2024-12-03' },
                { name: 'Delivery Receipt', type: 'POD', status: 'Received', date: '2024-12-28' }
            ],
            messages: [
                { from: 'Warehouse', text: 'All items received and checked. No discrepancies.', date: '2024-12-28T16:00:00Z' }
            ],
            co2Emissions: 2.4
        },
        {
            id: 'PO-2025-1798',
            poNumber: 'PO-2025-1798',
            buyerId: 'BUY002',
            supplierId: 'SUP003',
            forwarderId: 'FWD002',
            originAgentId: 'AGT001',
            destAgentId: 'AGT004',
            status: 'Confirmed',
            mode: 'Multimodal',
            incoterm: 'DDP',
            origin: 'Taipei, Taiwan',
            destination: 'Munich, Germany',
            cargoReadyDate: '2025-01-20',
            eta: '2025-02-05',
            etaConfidence: 'Estimated',
            vesselName: null,
            voyageNumber: null,
            containerNumber: null,
            blNumber: null,
            totalValue: 1250000,
            currency: 'USD',
            totalWeight: 8500,
            weightUnit: 'kg',
            dataQuality: 72,
            priority: 'Critical',
            notes: 'High-value chip manufacturing equipment',
            createdAt: '2025-01-05T09:00:00Z',
            updatedAt: '2025-01-08T07:45:00Z',
            milestones: [
                { name: 'PO Created', status: 'completed', date: '2025-01-05', actual: '2025-01-05' },
                { name: 'PO Confirmed', status: 'completed', date: '2025-01-07', actual: '2025-01-07' },
                { name: 'Production Started', status: 'pending', date: '2025-01-10', actual: null },
                { name: 'Cargo Ready', status: 'pending', date: '2025-01-20', actual: null },
                { name: 'Shipped', status: 'pending', date: '2025-01-22', actual: null },
                { name: 'Delivered', status: 'pending', date: '2025-02-05', actual: null }
            ],
            lineItems: [
                { sku: 'MACH-CNC-01', description: 'CNC Precision Machine', qty: 2, unitPrice: 450000, total: 900000 },
                { sku: 'MACH-ACC-10', description: 'Tool Head Set', qty: 10, unitPrice: 25000, total: 250000 },
                { sku: 'MACH-SFT-01', description: 'Control Software License', qty: 2, unitPrice: 50000, total: 100000 }
            ],
            documents: [
                { name: 'Proforma Invoice', type: 'Proforma', status: 'Received', date: '2025-01-06' },
                { name: 'Technical Specifications', type: 'Specs', status: 'Received', date: '2025-01-06' }
            ],
            messages: [
                { from: 'Supplier', text: 'Order confirmed. Production scheduling in progress.', date: '2025-01-07T14:00:00Z' }
            ],
            co2Emissions: null
        }
    ],

    // Suppliers
    suppliers: [
        { id: 'SUP001', name: 'Yantian Manufacturing Co.', country: 'China', rating: 4.5, onTimeRate: 94, avgLeadTime: 21, category: 'Electronics' },
        { id: 'SUP002', name: 'Global Parts Ltd', country: 'Singapore', rating: 4.8, onTimeRate: 97, avgLeadTime: 14, category: 'Automotive' },
        { id: 'SUP003', name: 'Taiwan Precision Industries', country: 'Taiwan', rating: 4.6, onTimeRate: 92, avgLeadTime: 18, category: 'Semiconductors' },
        { id: 'SUP004', name: 'Makoto Industries', country: 'Japan', rating: 4.9, onTimeRate: 98, avgLeadTime: 16, category: 'Precision Instruments' },
        { id: 'SUP005', name: 'Shanghai Export Corporation', country: 'China', rating: 4.2, onTimeRate: 88, avgLeadTime: 24, category: 'Textiles' },
        { id: 'SUP006', name: 'Vietnam Manufacturing', country: 'Vietnam', rating: 4.3, onTimeRate: 90, avgLeadTime: 20, category: 'Apparel' },
        { id: 'SUP007', name: 'Korea Electronics', country: 'South Korea', rating: 4.7, onTimeRate: 95, avgLeadTime: 15, category: 'Electronics' },
        { id: 'SUP008', name: 'India Textiles Ltd', country: 'India', rating: 4.1, onTimeRate: 85, avgLeadTime: 28, category: 'Textiles' }
    ],

    // Buyers
    buyers: [
        { id: 'BUY001', name: 'AGOHA TRENDS', country: 'USA', contact: 'Sarah Johnson', email: 'sarah@agoha.com' },
        { id: 'BUY002', name: 'HONDA MOTORS UK', country: 'UK', contact: 'James Smith', email: 'jsmith@honda.co.uk' },
        { id: 'BUY003', name: 'PURPPLE FASHIONS', country: 'Netherlands', contact: 'Anna van der Berg', email: 'anna@purpple.nl' },
        { id: 'BUY004', name: 'BERLIN TEXTILES GmbH', country: 'Germany', contact: 'Klaus Mueller', email: 'mueller@berlintex.de' },
        { id: 'BUY005', name: 'WEST COAST IMPORTS', country: 'USA', contact: 'Michael Chen', email: 'mchen@wcimports.com' }
    ],

    // Agents
    agents: [
        { id: 'AGT001', name: 'Orient Express Logistics', type: 'Origin', country: 'China', city: 'Shenzhen' },
        { id: 'AGT002', name: 'Pacific Gateway', type: 'Destination', country: 'USA', city: 'Los Angeles' },
        { id: 'AGT003', name: 'Singapore Freight Hub', type: 'Origin', country: 'Singapore', city: 'Singapore' },
        { id: 'AGT004', name: 'Euro Logistics Partners', type: 'Destination', country: 'UK', city: 'London' },
        { id: 'AGT005', name: 'Tokyo Cargo Services', type: 'Origin', country: 'Japan', city: 'Tokyo' },
        { id: 'AGT006', name: 'Hamburg Port Services', type: 'Destination', country: 'Germany', city: 'Hamburg' }
    ],

    // Forwarders
    forwarders: [
        { id: 'FWD001', name: 'Maersk Line', rating: 4.6, onTimeRate: 91, avgTransitTime: 22 },
        { id: 'FWD002', name: 'DHL Global Forwarding', rating: 4.7, onTimeRate: 94, avgTransitTime: 8 },
        { id: 'FWD003', name: 'ONE (Ocean Network Express)', rating: 4.5, onTimeRate: 89, avgTransitTime: 24 },
        { id: 'FWD004', name: 'Kuehne+Nagel', rating: 4.8, onTimeRate: 93, avgTransitTime: 18 },
        { id: 'FWD005', name: 'DB Schenker', rating: 4.4, onTimeRate: 87, avgTransitTime: 20 }
    ],

    // Alerts
    alerts: [
        {
            id: 'ALT001',
            type: 'Delay',
            priority: 'Critical',
            title: 'Vessel Delay - COSCO Shipping Aries',
            description: 'Vessel experiencing 2-day delay due to port congestion at Yantian. ETA impact on 3 shipments.',
            recommendation: 'Notify affected buyers immediately. Consider expedited customs clearance to recover time.',
            affectedOrders: ['PO-2025-1847'],
            status: 'Open',
            assignedTo: null,
            createdAt: '2025-01-08T06:00:00Z',
            acknowledgedAt: null
        },
        {
            id: 'ALT002',
            type: 'Missing Document',
            priority: 'High',
            title: 'Missing Certificate of Origin',
            description: 'COO not received for PO-2025-1847. Required for customs clearance.',
            recommendation: 'Request COO from supplier immediately. Document is required 5 days before port arrival.',
            affectedOrders: ['PO-2025-1847'],
            status: 'Open',
            assignedTo: 'Sarah Chen',
            createdAt: '2025-01-07T14:30:00Z',
            acknowledgedAt: '2025-01-07T15:00:00Z'
        },
        {
            id: 'ALT003',
            type: 'Milestone Overdue',
            priority: 'High',
            title: 'Production Milestone Overdue',
            description: 'PO-2025-1846 production completion overdue by 1 day.',
            recommendation: 'Contact supplier for status update. Assess impact on delivery timeline.',
            affectedOrders: ['PO-2025-1846'],
            status: 'Open',
            assignedTo: null,
            createdAt: '2025-01-08T08:00:00Z',
            acknowledgedAt: null
        },
        {
            id: 'ALT004',
            type: 'Port Congestion',
            priority: 'Medium',
            title: 'Rotterdam Port Congestion Alert',
            description: 'Increased vessel waiting times at Rotterdam. May affect arrivals scheduled for Feb 1-10.',
            recommendation: 'Monitor situation. Consider alternative discharge ports if delays exceed 3 days.',
            affectedOrders: ['PO-2025-1845'],
            status: 'Open',
            assignedTo: null,
            createdAt: '2025-01-07T10:00:00Z',
            acknowledgedAt: null
        },
        {
            id: 'ALT005',
            type: 'Compliance',
            priority: 'Medium',
            title: 'Supplier Performance Below Threshold',
            description: 'Shanghai Export Corporation OTD dropped to 85% (threshold: 90%).',
            recommendation: 'Schedule supplier review meeting. Consider backup suppliers for critical orders.',
            affectedOrders: ['PO-2025-1843'],
            status: 'Open',
            assignedTo: 'Mike Johnson',
            createdAt: '2025-01-06T09:00:00Z',
            acknowledgedAt: '2025-01-06T11:00:00Z'
        },
        {
            id: 'ALT006',
            type: 'Capacity',
            priority: 'Low',
            title: 'Partial Shipment Risk',
            description: 'PO-2025-1845 may require partial shipment due to vessel space constraints.',
            recommendation: 'Confirm full container availability with forwarder by Jan 9.',
            affectedOrders: ['PO-2025-1845'],
            status: 'Open',
            assignedTo: null,
            createdAt: '2025-01-05T16:00:00Z',
            acknowledgedAt: null
        },
        {
            id: 'ALT007',
            type: 'Data Mismatch',
            priority: 'Low',
            title: 'Weight Discrepancy Detected',
            description: 'Declared weight differs from booking weight by 8% for PO-2025-1844.',
            recommendation: 'Verify actual weight with origin agent before customs filing.',
            affectedOrders: ['PO-2025-1844'],
            status: 'Acknowledged',
            assignedTo: 'Lisa Wong',
            createdAt: '2025-01-04T12:00:00Z',
            acknowledgedAt: '2025-01-04T14:30:00Z'
        }
    ],

    // Analytics
    analytics: {
        otd: 91.5,
        otdTrend: 2.3,
        avgLeadTime: 18,
        documentCompliance: 94,
        exceptionRate: 8.2,
        supplierResponseTime: 4.2,
        co2PerShipment: 3.8,
        totalCO2: 285.6,
        co2ByMode: {
            sea: 62,
            air: 28,
            road: 10
        },
        monthlyOTD: [88, 89, 91, 90, 92, 91, 93, 91, 90, 92, 91, 92],
        monthlyCO2: [22, 24, 26, 23, 25, 24, 26, 25, 23, 24, 25, 24]
    },

    // Forecasting
    forecasting: {
        historical: [
            { month: 'Jan 2025', orders: 45, value: 2800000 },
            { month: 'Feb 2025', orders: 52, value: 3200000 },
            { month: 'Mar 2025', orders: 48, value: 2950000 },
            { month: 'Apr 2025', orders: 55, value: 3400000 },
            { month: 'May 2025', orders: 61, value: 3800000 },
            { month: 'Jun 2025', orders: 58, value: 3550000 },
            { month: 'Jul 2025', orders: 63, value: 3900000 },
            { month: 'Aug 2025', orders: 67, value: 4200000 },
            { month: 'Sep 2025', orders: 72, value: 4500000 },
            { month: 'Oct 2025', orders: 78, value: 4900000 },
            { month: 'Nov 2025', orders: 82, value: 5200000 },
            { month: 'Dec 2025', orders: 75, value: 4700000 }
        ],
        forecast: [
            { month: 'Jan 2026', orders: 48, value: 3000000, confidence: 'high' },
            { month: 'Feb 2026', orders: 55, value: 3400000, confidence: 'high' },
            { month: 'Mar 2026', orders: 52, value: 3200000, confidence: 'medium' },
            { month: 'Apr 2026', orders: 60, value: 3700000, confidence: 'medium' },
            { month: 'May 2026', orders: 65, value: 4100000, confidence: 'low' },
            { month: 'Jun 2026', orders: 62, value: 3850000, confidence: 'low' }
        ],
        bySupplier: [
            { supplierId: 'SUP001', q1Orders: 18, q1Value: 1200000, trend: 'up' },
            { supplierId: 'SUP002', q1Orders: 12, q1Value: 950000, trend: 'stable' },
            { supplierId: 'SUP003', q1Orders: 15, q1Value: 1100000, trend: 'up' },
            { supplierId: 'SUP004', q1Orders: 8, q1Value: 680000, trend: 'down' },
            { supplierId: 'SUP005', q1Orders: 10, q1Value: 720000, trend: 'stable' }
        ],
        capacity: {
            warehouseUtilization: 78,
            transportCapacity: 85,
            peakSeason: ['Sep', 'Oct', 'Nov']
        },
        seasonality: [0.85, 0.90, 0.88, 0.95, 1.05, 1.00, 1.08, 1.15, 1.25, 1.35, 1.42, 1.30]
    }
};

// ==========================================
// HELPER FUNCTIONS
// ==========================================

const ConexusHelpers = {
    // Get supplier by ID
    getSupplier(id) {
        return ConexusData.suppliers.find(s => s.id === id) || { name: 'Unknown Supplier' };
    },

    // Get buyer by ID
    getBuyer(id) {
        return ConexusData.buyers.find(b => b.id === id) || { name: 'Unknown Buyer' };
    },

    // Get agent by ID
    getAgent(id) {
        return ConexusData.agents.find(a => a.id === id) || { name: 'Unknown Agent' };
    },

    // Get forwarder by ID
    getForwarder(id) {
        return ConexusData.forwarders.find(f => f.id === id) || { name: 'Unknown Forwarder' };
    },

    // Get order by ID
    getOrder(id) {
        return ConexusData.orders.find(o => o.id === id);
    },

    // Get orders by status
    getOrdersByStatus(status) {
        if (status === 'all') return ConexusData.orders;
        return ConexusData.orders.filter(o => o.status === status);
    },

    // Get critical alerts count
    getCriticalAlertsCount() {
        return ConexusData.alerts.filter(a => a.priority === 'Critical' && a.status === 'Open').length;
    },

    // Get open alerts count
    getOpenAlertsCount() {
        return ConexusData.alerts.filter(a => a.status === 'Open').length;
    },

    // Get pipeline counts
    getPipelineCounts() {
        const counts = {};
        const statusMap = {
            'Placed': 'placed',
            'Confirmed': 'confirmed',
            'Production': 'production',
            'Cargo Ready': 'cargo-ready',
            'Shipped': 'shipped',
            'In Transit': 'in-transit',
            'Arrived': 'arrived',
            'Delivered': 'delivered',
            'On Hold': 'on-hold',
            'Cancelled': 'cancelled'
        };
        
        ConexusData.config.statuses.forEach(status => {
            const key = statusMap[status] || status.toLowerCase().replace(' ', '-');
            counts[key] = ConexusData.orders.filter(o => o.status === status).length;
        });
        
        return counts;
    },

    // Format currency
    formatCurrency(amount, currency = 'USD') {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: 0,
            maximumFractionDigits: 0
        }).format(amount);
    },

    // Format date
    formatDate(dateStr) {
        if (!dateStr) return '-';
        const date = new Date(dateStr);
        return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
    },

    // Get status color
    getStatusColor(status) {
        const colors = {
            'Placed': '#6366f1',
            'Confirmed': '#3b82f6',
            'Production': '#8b5cf6',
            'Cargo Ready': '#06b6d4',
            'Shipped': '#10b981',
            'In Transit': '#f59e0b',
            'Arrived': '#22c55e',
            'Delivered': '#059669',
            'On Hold': '#ef4444',
            'Cancelled': '#64748b'
        };
        return colors[status] || '#64748b';
    },

    // Get data quality level
    getDataQualityLevel(score) {
        if (score >= 90) return { label: 'Excellent', class: 'success', color: '#10b981' };
        if (score >= 70) return { label: 'Good', class: 'info', color: '#3b82f6' };
        if (score >= 50) return { label: 'Fair', class: 'warning', color: '#f59e0b' };
        return { label: 'Poor', class: 'error', color: '#ef4444' };
    },

    // Get confidence badge
    getConfidenceBadge(confidence) {
        const badges = {
            'Confirmed': { class: 'success', icon: '✓' },
            'Estimated': { class: 'warning', icon: '~' },
            'Unknown': { class: 'muted', icon: '?' }
        };
        return badges[confidence] || badges['Unknown'];
    },

    // Search orders
    searchOrders(query) {
        if (!query) return ConexusData.orders;
        const q = query.toLowerCase();
        return ConexusData.orders.filter(o => 
            o.id.toLowerCase().includes(q) ||
            o.poNumber.toLowerCase().includes(q) ||
            (o.containerNumber && o.containerNumber.toLowerCase().includes(q)) ||
            (o.blNumber && o.blNumber.toLowerCase().includes(q)) ||
            (o.vesselName && o.vesselName.toLowerCase().includes(q)) ||
            this.getSupplier(o.supplierId).name.toLowerCase().includes(q) ||
            this.getBuyer(o.buyerId).name.toLowerCase().includes(q)
        );
    },

    // Generate order ID
    generateOrderId() {
        const year = new Date().getFullYear();
        const num = Math.floor(Math.random() * 9000) + 1000;
        return `PO-${year}-${num}`;
    },

    // Calculate OTD
    calculateOTD() {
        const delivered = ConexusData.orders.filter(o => o.status === 'Delivered');
        if (delivered.length === 0) return 0;
        const onTime = delivered.filter(o => {
            const eta = new Date(o.eta);
            const actual = o.milestones.find(m => m.name === 'Delivered')?.actual;
            if (!actual) return true;
            return new Date(actual) <= eta;
        });
        return Math.round((onTime.length / delivered.length) * 100);
    },

    // Get alert priority color
    getAlertPriorityColor(priority) {
        const colors = {
            'Critical': '#ef4444',
            'High': '#f59e0b',
            'Medium': '#3b82f6',
            'Low': '#64748b'
        };
        return colors[priority] || '#64748b';
    },

    // Get alerts by priority
    getAlertsByPriority(priority) {
        return ConexusData.alerts.filter(a => a.priority === priority && a.status === 'Open');
    },

    // Get alerts by type
    getAlertsByType(type) {
        return ConexusData.alerts.filter(a => a.type === type && a.status === 'Open');
    }
};

// Initialize on load
document.addEventListener('DOMContentLoaded', function() {
    console.log('Conexus Data loaded:', ConexusData.orders.length, 'orders');
});
